/**
 * Service Worker - gestion des messages entre content scripts et popup.
 */

// Compteur de produits ajoutes dans cette session
let productsAdded = 0;

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === "SCRAPE_PRODUCT") {
    handleScrape(message.data)
      .then((result) => sendResponse({ success: true, ...result }))
      .catch((err) => sendResponse({ success: false, error: err.message }));
    return true;
  }

  if (message.type === "GET_STATUS") {
    handleGetStatus()
      .then((status) => sendResponse(status))
      .catch(() => sendResponse({ connected: false }));
    return true;
  }

  if (message.type === "CONNECT") {
    handleConnect(message.url, message.token)
      .then((result) => sendResponse({ success: true, ...result }))
      .catch((err) => sendResponse({ success: false, error: err.message }));
    return true;
  }

  if (message.type === "DISCONNECT") {
    handleDisconnect()
      .then(() => sendResponse({ success: true }))
      .catch(() => sendResponse({ success: false }));
    return true;
  }

  if (message.type === "GET_PRODUCTS_COUNT") {
    sendResponse({ count: productsAdded });
    return false;
  }

  // QR scan flow: popup demande le scan
  if (message.type === "START_SCAN") {
    startScanFlow()
      .then(() => sendResponse({ success: true }))
      .catch((err) => sendResponse({ success: false, error: err.message }));
    return true;
  }

  // QR decode result from scanner overlay
  if (message.type === "QR_SCANNED") {
    handleQRScanned(message.data)
      .then((result) => sendResponse(result))
      .catch((err) => sendResponse({ success: false, error: err.message }));
    return true;
  }
});


// ============================================
// QR Scan Flow
// ============================================
async function startScanFlow() {
  // 1. Get active tab
  const [tab] = await chrome.tabs.query({
    active: true,
    currentWindow: true,
  });
  if (!tab) throw new Error("Aucun onglet actif");

  // 2. Capture visible tab BEFORE injecting overlay
  const dataUrl = await chrome.tabs.captureVisibleTab(tab.windowId, {
    format: "png",
  });

  // 3. Inject jsQR library + scanner overlay script
  await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    files: ["lib/jsQR.js", "content/scanner-overlay.js"],
  });

  // 4. Send screenshot to the overlay
  await chrome.tabs.sendMessage(tab.id, {
    type: "SHOW_SCANNER",
    dataUrl: dataUrl,
  });
}

async function handleQRScanned(raw) {
  let url, token;
  try {
    const data = JSON.parse(raw);
    url = data.url;
    token = data.token;
  } catch {
    return { success: false, error: "QR code invalide (format JSON attendu)" };
  }

  if (!url || !token) {
    return { success: false, error: "QR code incomplet (url et token requis)" };
  }

  try {
    await handleConnect(url, token);
    return { success: true };
  } catch (err) {
    return { success: false, error: err.message };
  }
}


// ============================================
// Scrape
// ============================================
async function handleScrape(productData) {
  const session = await getSession();
  if (!session) throw new Error("Non connecte");

  const response = await fetch(`${session.url}/api/extensions/scrape`, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({
      token: session.token,
      product_data: productData,
    }),
  });

  if (!response.ok) {
    const error = await response.json().catch(() => ({}));
    throw new Error(error.detail || `Erreur ${response.status}`);
  }

  const result = await response.json();
  productsAdded++;

  // Mettre a jour le badge
  chrome.action.setBadgeText({ text: String(productsAdded) });
  chrome.action.setBadgeBackgroundColor({ color: "#7c3aed" });

  return result;
}


// ============================================
// Connection
// ============================================
async function handleGetStatus() {
  const session = await getSession();
  if (!session) return { connected: false };

  try {
    const response = await fetch(
      `${session.url}/api/extensions/sessions/${session.token}`
    );
    if (!response.ok) return { connected: false };
    const data = await response.json();
    return {
      connected: data.status === "connected",
      status: data.status,
      csvFileId: data.csv_file_id,
      token: session.token,
      url: session.url,
    };
  } catch {
    return { connected: false };
  }
}

async function handleConnect(url, token) {
  const response = await fetch(`${url}/api/extensions/connect`, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({ token }),
  });

  if (!response.ok) {
    const error = await response.json().catch(() => ({}));
    throw new Error(error.detail || `Erreur ${response.status}`);
  }

  // Sauvegarder la session
  await chrome.storage.local.set({
    dropping_session: { url, token },
  });

  productsAdded = 0;
  chrome.action.setBadgeText({ text: "" });

  return response.json();
}

async function handleDisconnect() {
  await chrome.storage.local.remove("dropping_session");
  productsAdded = 0;
  chrome.action.setBadgeText({ text: "" });
}

async function getSession() {
  const data = await chrome.storage.local.get("dropping_session");
  return data.dropping_session || null;
}
