/**
 * Content Script - Detecte les pages produit et injecte un bouton flottant (FAB).
 * Extraction multi-niveaux: JSON-LD -> Embedded JSON -> DOM
 * Interception MTOP API pour capturer le vrai skuId numerique.
 */

(function () {
  // Eviter les doubles injections
  if (document.getElementById("dropping-fab")) return;

  const hostname = window.location.hostname;
  let siteName = null;

  if (hostname.includes("aliexpress.com")) {
    siteName = "aliexpress";
  } else if (hostname.includes("amazon.")) {
    siteName = "amazon";
  }

  if (!siteName) return;

  // L'interception MTOP est geree par mtop-interceptor.js (injecte a document_start)
  // Les donnees MTOP sont stockees sur <html> via data-dropping-sku-id et data-dropping-sku-price

  // Creer le FAB avec des methodes DOM safe (pas de innerHTML)
  const container = document.createElement("div");
  container.id = "dropping-fab";

  // Bouton principal
  const btn = document.createElement("div");
  btn.id = "dropping-fab-btn";
  btn.title = "Scraper avec Dropping";
  Object.assign(btn.style, {
    position: "fixed",
    bottom: "24px",
    right: "24px",
    zIndex: "999999",
    width: "56px",
    height: "56px",
    borderRadius: "50%",
    background: "linear-gradient(135deg, #7c3aed, #6d28d9)",
    color: "white",
    display: "flex",
    alignItems: "center",
    justifyContent: "center",
    cursor: "pointer",
    boxShadow: "0 4px 12px rgba(124, 58, 237, 0.4)",
    transition: "all 0.2s ease",
    fontSize: "20px",
  });

  // Icone SVG (plus)
  const svg = document.createElementNS("http://www.w3.org/2000/svg", "svg");
  svg.setAttribute("width", "24");
  svg.setAttribute("height", "24");
  svg.setAttribute("viewBox", "0 0 24 24");
  svg.setAttribute("fill", "none");
  svg.setAttribute("stroke", "currentColor");
  svg.setAttribute("stroke-width", "2");
  const path = document.createElementNS("http://www.w3.org/2000/svg", "path");
  path.setAttribute("d", "M12 5v14M5 12h14");
  svg.appendChild(path);
  btn.appendChild(svg);

  // Status indicator
  const statusEl = document.createElement("div");
  statusEl.id = "dropping-fab-status";
  Object.assign(statusEl.style, {
    position: "fixed",
    bottom: "88px",
    right: "24px",
    zIndex: "999999",
    background: "#18181b",
    color: "#a1a1aa",
    padding: "8px 16px",
    borderRadius: "8px",
    fontSize: "13px",
    fontFamily: "-apple-system, system-ui, sans-serif",
    boxShadow: "0 4px 12px rgba(0,0,0,0.3)",
    display: "none",
    border: "1px solid #27272a",
    maxWidth: "280px",
  });

  container.appendChild(btn);
  container.appendChild(statusEl);
  document.body.appendChild(container);

  // Hover effect
  btn.addEventListener("mouseenter", () => {
    btn.style.transform = "scale(1.1)";
    btn.style.boxShadow = "0 6px 20px rgba(124, 58, 237, 0.5)";
  });
  btn.addEventListener("mouseleave", () => {
    btn.style.transform = "scale(1)";
    btn.style.boxShadow = "0 4px 12px rgba(124, 58, 237, 0.4)";
  });

  // Click handler
  btn.addEventListener("click", async () => {
    showStatus("Extraction en cours...", "loading");

    try {
      let productData;
      if (siteName === "aliexpress") {
        productData = extractAliExpress();
      } else if (siteName === "amazon") {
        productData = extractAmazon();
      }

      if (!productData || !productData.title) {
        showStatus("Impossible d'extraire les donnees de cette page", "error");
        return;
      }

      // Envoyer au service worker
      const response = await chrome.runtime.sendMessage({
        type: "SCRAPE_PRODUCT",
        data: productData,
      });

      if (response.success) {
        const shortTitle = productData.title.length > 40
          ? productData.title.slice(0, 40) + "..."
          : productData.title;
        showStatus("Ajoute: " + shortTitle, "success");
        btn.style.background = "linear-gradient(135deg, #22c55e, #16a34a)";
        setTimeout(() => {
          btn.style.background = "linear-gradient(135deg, #7c3aed, #6d28d9)";
        }, 1500);
      } else {
        showStatus("Erreur: " + response.error, "error");
      }
    } catch (err) {
      showStatus("Erreur: " + err.message, "error");
    }
  });

  function showStatus(message, type) {
    statusEl.style.display = "block";
    statusEl.textContent = message;

    if (type === "success") {
      statusEl.style.borderColor = "#22c55e";
      statusEl.style.color = "#86efac";
    } else if (type === "error") {
      statusEl.style.borderColor = "#ef4444";
      statusEl.style.color = "#fca5a5";
    } else {
      statusEl.style.borderColor = "#7c3aed";
      statusEl.style.color = "#c4b5fd";
    }

    setTimeout(() => {
      statusEl.style.display = "none";
    }, 4000);
  }

  // ============================================
  // Helpers
  // ============================================
  function getText(sel) {
    const el = document.querySelector(sel);
    return el ? el.textContent.trim() : "";
  }

  function getAttr(sel, attr) {
    const el = document.querySelector(sel);
    return el ? el.getAttribute(attr) || "" : "";
  }

  /** Recherche recursive d'une cle dans un objet imbrique. */
  function deepFind(data, key, maxDepth) {
    if (maxDepth === undefined) maxDepth = 10;
    if (maxDepth <= 0 || !data) return null;
    if (typeof data === "object" && !Array.isArray(data)) {
      if (key in data) return data[key];
      for (const k of Object.keys(data)) {
        const result = deepFind(data[k], key, maxDepth - 1);
        if (result !== null) return result;
      }
    }
    if (Array.isArray(data)) {
      for (const item of data) {
        const result = deepFind(item, key, maxDepth - 1);
        if (result !== null) return result;
      }
    }
    return null;
  }

  /** Cherche la premiere valeur non-vide pour une liste de cles. */
  function deepFindValue(data, keys) {
    for (const key of keys) {
      const val = deepFind(data, key);
      if (val !== null && val !== undefined && val !== "") return val;
    }
    return null;
  }

  /** Extraire un objet JSON depuis les scripts de la page. */
  function extractEmbeddedJson(patterns) {
    const scripts = document.querySelectorAll("script:not([src])");
    for (const script of scripts) {
      const text = script.textContent || "";
      if (text.length < 500) continue;
      for (const pattern of patterns) {
        const match = text.match(pattern);
        if (!match) continue;
        const start = match.index + match[0].length;
        let depth = 0;
        let inString = false;
        let escape = false;
        let jsonStart = -1;
        for (let i = start; i < text.length && i < start + 500000; i++) {
          const ch = text[i];
          if (escape) { escape = false; continue; }
          if (ch === "\\") { escape = true; continue; }
          if (ch === '"' && !escape) { inString = !inString; continue; }
          if (inString) continue;
          if (ch === "{" || ch === "[") {
            if (depth === 0) jsonStart = i;
            depth++;
          } else if (ch === "}" || ch === "]") {
            depth--;
            if (depth === 0 && jsonStart >= 0) {
              try {
                return JSON.parse(text.slice(jsonStart, i + 1));
              } catch { break; }
            }
          }
        }
      }
    }
    return null;
  }

  /** Normaliser l'URL AliExpress: retirer sous-domaine + ajouter pdp_ext_f pour deep-linking variante. */
  function normalizeAliExpressUrl(skuId) {
    const parsed = new URL(window.location.href);
    // Retirer le sous-domaine (fr.aliexpress.com -> aliexpress.com)
    const hostParts = parsed.hostname.split(".");
    const baseDomain = hostParts.length > 2
      ? hostParts.slice(-2).join(".")
      : parsed.hostname;
    let url = parsed.protocol + "//" + baseDomain + parsed.pathname;
    // Ajouter pdp_ext_f avec sku_id pour pointer directement sur la variante
    if (skuId) {
      url += "?pdp_ext_f=" + encodeURIComponent(JSON.stringify({ sku_id: String(skuId) }));
    }
    return url;
  }

  /** Nettoyer une URL d'image AliExpress (remplacer suffixe de redimensionnement par taille standard).
   *  Le CDN AliExpress ne sert pas les images sans suffixe de taille.
   *  Ex: image.jpg_220x220q75.jpg_.avif -> image.jpg_640x640.jpg
   */
  function cleanAliImageUrl(src) {
    if (!src || src.includes("data:")) return "";
    // Remplacer le suffixe de resize par une taille plus grande
    src = src.replace(/(\.(jpg|jpeg|png|webp))_\d+x\d+[^/]*$/i, "$1_640x640.jpg");
    // Assurer protocole
    if (src.startsWith("//")) src = "https:" + src;
    if (!src.startsWith("http")) return "";
    return src;
  }

  /** Extraire le skuAttr depuis la selection de variante sur la page.
   *  Format AliExpress: "propId:valueId" separes par ";", ex: "14:193;5:100014065"
   *  Ce format est utilise dans les URLs AliExpress pour pointer sur la bonne variante.
   */
  function getSelectedSkuAttr() {
    const selectedSkuEls = document.querySelectorAll(
      "[class*='sku-item--selected'][data-sku-col]"
    );
    if (selectedSkuEls.length === 0) return "";

    // data-sku-col="14-193" -> "14:193"
    const parts = Array.from(selectedSkuEls)
      .map((el) => {
        const col = el.getAttribute("data-sku-col") || "";
        const [propId, valueId] = col.split("-");
        return propId && valueId ? propId + ":" + valueId : "";
      })
      .filter(Boolean);
    return parts.join(";");
  }

  /** Extraire le skuId numerique (long ID) depuis l'interception MTOP, l'URL ou les donnees JSON. */
  function getSkuId(modules) {
    // 1. PRIORITE: depuis l'interception MTOP (data-attribute sur <html>)
    // C'est la source la plus fiable - contient le vrai selectedSkuId
    const mtopSkuId = document.documentElement.getAttribute("data-dropping-sku-id");
    if (mtopSkuId) return String(mtopSkuId);

    // 2. Depuis l'URL actuelle (si deja present)
    const urlParams = new URLSearchParams(window.location.search);
    const fromUrl = urlParams.get("skuId") || urlParams.get("sku_id") || urlParams.get("skuid") || "";
    if (fromUrl) return String(fromUrl);

    // 3. Depuis pdp_ext_f dans l'URL
    const extF = urlParams.get("pdp_ext_f");
    if (extF) {
      try {
        const parsed = JSON.parse(extF);
        if (parsed.sku_id) return String(parsed.sku_id);
      } catch { /* skip */ }
    }

    // 4. Ancien format data-sku-id (certaines pages)
    const selectedSkuId = document.querySelector(
      "[class*='sku-item--selected'] [data-sku-id], " +
      "[data-sku-id][class*='active']"
    );
    if (selectedSkuId) {
      const val = selectedSkuId.getAttribute("data-sku-id");
      if (val) return String(val);
    }

    // 5. Depuis __NEXT_DATA__ (Next.js)
    try {
      const nextDataEl = document.getElementById("__NEXT_DATA__");
      if (nextDataEl) {
        const nextData = JSON.parse(nextDataEl.textContent);
        const skuFromNext = deepFindValue(nextData, [
          "skuId", "selectedSkuId", "sku_id",
        ]);
        if (skuFromNext) return String(skuFromNext);
      }
    } catch { /* skip */ }

    // 6. Depuis les donnees JSON (skuModule)
    if (modules) {
      const skuMod = deepFind(modules, "skuModule") || deepFind(modules, "sku");
      if (skuMod && typeof skuMod === "object") {
        const priceList = skuMod.skuPriceList || [];
        if (Array.isArray(priceList) && priceList.length > 0) {
          return String(priceList[0].skuId || "");
        }
      }
      const skuId = deepFindValue(modules, ["skuId", "selectedSkuId"]);
      if (skuId) return String(skuId);
    }

    return "";
  }

  /** Extraire les infos DOM communes pour AliExpress (seller, shipping, etc.) */
  function extractAliExpressDomExtras() {
    // Seller / Store name
    let seller =
      getText("[class*='store-detail--storeName']") ||
      getText("[data-pl='store-name']") ||
      getText("[class*='store-info--name'] a") ||
      getText("[class*='storeName']") ||
      getText("[class*='StoreName'] a") ||
      getText(".store-header-name");
    // Nettoyer le nom du store
    seller = seller.replace(/\(Commerçant\)/gi, "").replace(/\(Merchant\)/gi, "").trim();

    // Shipping - extraire le premier prix/texte de livraison
    let shipping = "";
    const shippingItems = document.querySelectorAll("[class*='shipping--content'] .dynamic-shipping-line");
    for (const item of shippingItems) {
      const text = item.textContent.trim();
      if (text.match(/livraison|shipping|delivery|gratuit|free/i) && text.match(/[\d.,€$]/)) {
        shipping = text;
        break;
      }
    }
    if (!shipping) {
      shipping = getText("[class*='shipping--content'] [class*='titleLayout']") ||
        getText("[class*='Shipping'] span") ||
        getText(".product-shipping-info");
    }

    // Availability / stock
    const availability =
      getText("[class*='quantity--info']") ||
      getText("[class*='Quantity'] span") ||
      getText(".product-quantity-tip");

    // Category - breadcrumbs ou liens category
    let category = "";
    const breadcrumbEl = document.querySelector("[class*='breadcrumb']");
    if (breadcrumbEl) {
      const links = breadcrumbEl.querySelectorAll("a");
      if (links.length > 0) {
        category = links[links.length - 1].textContent.trim();
      }
    }
    if (!category) {
      const catLinks = document.querySelectorAll("[class*='Categoey'] a, [class*='Category'] a");
      if (catLinks.length > 0) {
        category = catLinks[catLinks.length - 1].textContent.trim();
      }
    }

    // Selected variant info (taille, couleur selectionnees)
    // Methode 1: lire le texte du sku-item--title qui contient deja "Taille: XL"
    const selectedVariants = [];
    const skuProps = document.querySelectorAll("[class*='sku-item--property']");
    for (const prop of skuProps) {
      // Chercher l'element selectionne dans cette propriete
      const selectedEl = prop.querySelector("[class*='sku-item--selected']");
      if (!selectedEl) continue;

      // Nom de la propriete depuis le titre
      const titleEl = prop.querySelector("[class*='sku-item--title']");
      const propName = titleEl
        ? titleEl.textContent.trim().replace(/\u00a0/g, " ").replace(/:\s*.*$/, "")
        : "";

      // Valeur selectionnee: title attr > texte du span > alt de l'img
      let value = selectedEl.getAttribute("title") || "";
      if (!value) {
        const textSpan = selectedEl.querySelector("span");
        if (textSpan) value = textSpan.textContent.trim();
      }
      if (!value) {
        const img = selectedEl.querySelector("img");
        if (img) value = img.getAttribute("alt") || "";
      }

      if (propName && value) {
        selectedVariants.push(propName + ": " + value);
      } else if (value) {
        selectedVariants.push(value);
      }
    }
    const variantLabel = selectedVariants.join(" / ");

    // Image de la variante selectionnee via data-sku-col
    // skuAttr = "5:100014065;14:193" -> on prend la partie couleur (avec image)
    // et on cherche [data-sku-col="14-193"] img
    let variantImageUrl = "";
    const skuAttrForImg = getSelectedSkuAttr();
    if (skuAttrForImg) {
      const parts = skuAttrForImg.split(";");
      for (const part of parts) {
        const skuCol = part.replace(":", "-"); // "14:193" -> "14-193"
        const imgEl = document.querySelector(
          "[data-sku-col=\"" + skuCol + "\"] img"
        );
        if (imgEl) {
          variantImageUrl = cleanAliImageUrl(imgEl.src || imgEl.dataset.src || "");
          break; // Premiere image trouvee = couleur selectionnee
        }
      }
    }
    // Fallback: chercher directement l'element selectionne avec image
    if (!variantImageUrl) {
      const selectedImg = document.querySelector(
        "[class*='sku-item--selected'][class*='sku-item--image'] img"
      );
      if (selectedImg) {
        variantImageUrl = cleanAliImageUrl(selectedImg.src || selectedImg.dataset.src || "");
      }
    }

    // Collecter TOUTES les images des variantes couleur (pour le champ images)
    const variantImages = [];
    const allImageItems = document.querySelectorAll("[class*='sku-item--image'] img");
    for (const img of allImageItems) {
      const src = cleanAliImageUrl(img.src || img.dataset.src || "");
      if (src && !variantImages.includes(src)) {
        variantImages.push(src);
      }
    }

    // Variantes count
    const variantEls = document.querySelectorAll(
      "[class*='sku-item--text'], [class*='sku-item--image']"
    );
    const variantsCount = String(variantEls.length || 0);

    // Store positive feedback
    const storeDesc = getText("[class*='store-info--desc']");

    return {
      seller,
      shipping,
      availability,
      category,
      variantLabel,
      variantImageUrl,
      variantImages,
      variantsCount,
      storeDesc,
    };
  }

  // ============================================
  // AliExpress - Combiner JSON-LD/embedded + DOM extras
  // ============================================
  function extractAliExpress() {
    // Extraire les infos DOM (toujours disponibles meme en CSR)
    const extras = extractAliExpressDomExtras();

    // Essayer JSON-LD d'abord (donnees de base fiables)
    let result = extractAliExpressJsonLd();

    // Si pas de JSON-LD, essayer embedded JSON
    if (!result || !result.title) {
      result = extractAliExpressEmbeddedJson();
    }

    // Si toujours rien, fallback DOM complet
    if (!result || !result.title) {
      result = extractAliExpressDom();
    }

    if (!result) return null;

    // Enrichir avec les extras DOM pour les champs vides
    if (!result.seller) result.seller = extras.seller;
    if (!result.shipping) result.shipping = extras.shipping;
    if (!result.availability) result.availability = extras.availability;
    if (!result.category) result.category = extras.category;
    if (extras.variantsCount !== "0") result.variants_count = extras.variantsCount;

    // Si variante selectionnee, utiliser son image comme image principale
    if (extras.variantImageUrl) {
      result.image_url = extras.variantImageUrl;
    }

    // Utiliser les images des variantes couleur (fiables, CDN correct)
    if (extras.variantImages && extras.variantImages.length > 0) {
      result.images = extras.variantImages.join("|");
    }

    // Ajouter le label de variante
    if (extras.variantLabel) {
      result.variant = extras.variantLabel;
    }

    return result;
  }

  /** Lire le prix de la variante selectionnee depuis l'interception MTOP. */
  function getMtopSkuPrice() {
    return document.documentElement.getAttribute("data-dropping-sku-price") || "";
  }

  function extractAliExpressJsonLd() {
    const scripts = document.querySelectorAll('script[type="application/ld+json"]');
    for (const script of scripts) {
      try {
        const ld = JSON.parse(script.textContent);
        const items = Array.isArray(ld) ? ld : [ld];
        for (const item of items) {
          if (item["@type"] !== "Product") continue;

          const title = item.name || "";
          if (!title) continue;

          let image = item.image || "";
          if (Array.isArray(image)) image = image[0] || "";

          let price = "";
          let currency = "USD";
          let offers = item.offers || {};
          if (Array.isArray(offers)) offers = offers[0] || {};
          if (typeof offers === "object") {
            price = String(offers.price || offers.lowPrice || "");
            currency = offers.priceCurrency || "USD";
          }

          // Utiliser le prix MTOP si disponible (plus precis pour la variante selectionnee)
          const mtopPrice = getMtopSkuPrice();
          if (mtopPrice) price = mtopPrice;

          let rating = "";
          let reviewsCount = "";
          const agg = item.aggregateRating || {};
          if (typeof agg === "object") {
            rating = String(agg.ratingValue || "");
            reviewsCount = String(agg.reviewCount || agg.ratingCount || "");
          }

          const brand = typeof item.brand === "object"
            ? (item.brand.name || "")
            : String(item.brand || "");

          // Extraire product_id depuis l'URL
          const idMatch = window.location.pathname.match(/\/item\/(\d+)\.html/);
          const productId = idMatch ? idMatch[1] : String(item.sku || item.productID || "");

          // SKU ID (depuis MTOP en priorite)
          const skuAttr = getSelectedSkuAttr();
          const skuId = getSkuId(null);

          return {
            title,
            price: price.replace(/[^\d.,]/g, ""),
            currency,
            url: normalizeAliExpressUrl(skuId),
            product_id: productId,
            sku_id: skuId || skuAttr,
            image_url: typeof image === "string" ? image : "",
            images: (Array.isArray(item.image) ? item.image : [image]).filter(Boolean).join("|"),
            rating,
            reviews_count: reviewsCount,
            description: String(item.description || "").slice(0, 500),
            category: "",
            brand,
            seller: "",
            shipping: "",
            availability: "",
            variants_count: String(Array.isArray(item.offers) ? item.offers.length : 0),
          };
        }
      } catch { /* skip */ }
    }
    return null;
  }

  function extractAliExpressEmbeddedJson() {
    // Essayer d'abord window._d_c_.DCData (nouveau format CSR)
    let data = null;
    try {
      if (window._d_c_ && window._d_c_.DCData) {
        data = window._d_c_.DCData;
      }
    } catch { /* skip */ }

    // Sinon chercher dans les scripts
    if (!data) {
      data = extractEmbeddedJson([
        /window\.runParams\s*=\s*/,
        /window\.__INIT_DATA__\s*=\s*/,
        /window\._d_c_\.DCData\s*=\s*/,
        /window\.\w+Data\s*=\s*/,
      ]);
    }
    if (!data) return null;

    const modules = data.data || data;

    // titleModule
    let title = "";
    const titleMod = deepFind(modules, "titleModule") || deepFind(modules, "title");
    if (titleMod && typeof titleMod === "object") {
      title = titleMod.subject || titleMod.title || titleMod.text || "";
    }

    // priceModule
    let price = "";
    let currency = "USD";
    const priceMod = deepFind(modules, "priceModule") || deepFind(modules, "price");
    if (priceMod && typeof priceMod === "object") {
      const priceRaw = priceMod.formatedActivityPrice
        || priceMod.formatedPrice
        || priceMod.minPrice
        || (priceMod.minAmount && priceMod.minAmount.value)
        || "";
      price = String(priceRaw).replace(/[^\d.,]/g, "");
      currency = priceMod.currencyCode || "USD";
    }

    // imageModule
    let images = [];
    let imageUrl = "";
    const imageMod = deepFind(modules, "imageModule") || deepFind(modules, "images");
    if (imageMod && typeof imageMod === "object") {
      const imgList = imageMod.imagePathList || imageMod.images || [];
      for (const img of imgList) {
        if (typeof img === "string") {
          images.push(img.startsWith("http") ? img : "https:" + img);
        }
      }
      imageUrl = images[0] || "";
    }

    // storeModule
    let seller = "";
    const storeMod = deepFind(modules, "storeModule") || deepFind(modules, "store");
    if (storeMod && typeof storeMod === "object") {
      seller = storeMod.storeName || storeMod.name || "";
    }

    // feedbackModule
    let rating = "";
    let reviewsCount = "";
    const feedbackMod = deepFind(modules, "feedbackModule") || deepFind(modules, "feedback");
    if (feedbackMod && typeof feedbackMod === "object") {
      rating = String(feedbackMod.averageStar || feedbackMod.evarageStar || "");
      reviewsCount = String(feedbackMod.totalValidNum || feedbackMod.reviewCount || "");
    }

    // shippingModule
    let shipping = "";
    const shippingMod = deepFind(modules, "shippingModule") || deepFind(modules, "shipping");
    if (shippingMod && typeof shippingMod === "object") {
      let freight = shippingMod.freightAmount || "";
      if (typeof freight === "object") freight = freight.formatedAmount || "";
      shipping = String(freight);
    }

    // tradeModule (commandes)
    let orders = "";
    const tradeMod = deepFind(modules, "tradeModule");
    if (tradeMod && typeof tradeMod === "object") {
      orders = String(tradeMod.tradeCount || tradeMod.formatTradeCount || "");
    }

    // skuModule (variantes)
    let variantsCount = "0";
    const skuMod = deepFind(modules, "skuModule") || deepFind(modules, "sku");
    if (skuMod && typeof skuMod === "object") {
      const skuList = skuMod.productSKUPropertyList || skuMod.skuPriceList || [];
      variantsCount = String(Array.isArray(skuList) ? skuList.length : 0);
    }

    // Fallback: recherche a plat
    if (!title) {
      title = String(deepFindValue(modules, ["subject", "productTitle", "title", "name"]) || "");
    }
    if (!price) {
      const p = deepFindValue(modules, ["formatedActivityPrice", "formatedPrice", "minPrice", "price"]);
      if (p) price = String(p).replace(/[^\d.,]/g, "");
    }
    if (!imageUrl) {
      const imgVal = deepFindValue(modules, ["imagePathList", "images"]);
      if (Array.isArray(imgVal) && imgVal.length > 0) {
        images = imgVal
          .filter((i) => typeof i === "string")
          .map((i) => (i.startsWith("http") ? i : "https:" + i));
        imageUrl = images[0] || "";
      }
    }

    if (!title) return null;

    // Description + categorie (DOM fallback)
    const descEl = document.querySelector(
      ".product-description, [class*='Description'], .detail-desc-decorate-richtext"
    );
    const description = descEl ? descEl.textContent.trim().slice(0, 500) : "";

    const breadcrumbs = document.querySelectorAll("[class*='Breadcrumb'] a, .breadcrumb a");
    const category = breadcrumbs.length > 0
      ? breadcrumbs[breadcrumbs.length - 1].textContent.trim()
      : "";

    // Extraire product_id depuis l'URL
    const idMatch = window.location.pathname.match(/\/item\/(\d+)\.html/);
    const productId = idMatch ? idMatch[1] : String(
      deepFindValue(modules, ["productId", "itemId", "id"]) || ""
    );

    // Utiliser le prix MTOP si disponible
    const mtopPrice = getMtopSkuPrice();
    if (mtopPrice) price = mtopPrice;

    // SKU ID
    const skuAttr = getSelectedSkuAttr();
    const skuId = getSkuId(modules);

    return {
      title,
      price,
      currency,
      url: normalizeAliExpressUrl(skuId),
      product_id: productId,
      sku_id: skuId || skuAttr,
      image_url: imageUrl,
      images: images.join("|"),
      rating,
      reviews_count: reviewsCount,
      description,
      category,
      brand: "",
      seller,
      shipping,
      availability: orders ? orders + " commandes" : "",
      variants_count: variantsCount,
    };
  }

  function extractAliExpressDom() {
    // Titre - nouveau format AliExpress (h1 ou h2)
    const title =
      getText("h1[data-pl='product-title']") ||
      getText("[class*='title--title']") ||
      getText("h1.product-title-text") ||
      getText("h1") ||
      getText("h2[class*='title']") ||
      getAttr("meta[property='og:title']", "content");

    // Prix - nouveau format
    let price =
      getText("[class*='price--current']") ||
      getText("[class*='price-default--current']") ||
      getText(".product-price-current .price-current-value") ||
      getText(".product-price-value") ||
      getText("[class*='Price_Price'] span") ||
      getText(".uniform-banner-box-price");
    price = price.replace(/[^\d.,]/g, "");

    // Currency depuis le symbole dans le prix
    const priceText =
      getText("[class*='price--current']") ||
      getText("[class*='price-default--current']");
    let currency = "USD";
    if (priceText.includes("€")) currency = "EUR";
    else if (priceText.includes("£")) currency = "GBP";
    else if (priceText.includes("$")) currency = "USD";

    // Images depuis DCData ou DOM
    let images = [];
    let imageUrl = "";
    try {
      if (window._d_c_ && window._d_c_.DCData && window._d_c_.DCData.imagePathList) {
        images = window._d_c_.DCData.imagePathList
          .filter((i) => typeof i === "string")
          .map((i) => i.startsWith("http") ? i : "https:" + i);
      }
    } catch { /* skip */ }

    if (images.length === 0) {
      const imageEls = document.querySelectorAll(
        "[class*='slider--img'] img, .images-view-item img, [class*='Gallery'] img, [class*='magnifier'] img"
      );
      images = Array.from(imageEls)
        .map((img) => img.src || img.dataset.src || "")
        .filter((src) => src && !src.includes("data:") && !src.includes("placeholder"))
        .map((src) => src.startsWith("http") ? src : "https:" + src);
    }
    imageUrl = images[0] || getAttr("meta[property='og:image']", "content");

    const rating =
      getText("[class*='EvaluationCard'] strong") ||
      getText(".overview-rating-average");

    const reviews =
      getText("[class*='EvaluationCard'] a") ||
      getText(".product-reviewer-reviews");

    const descEl = document.querySelector(
      ".product-description, [class*='Description'], .detail-desc-decorate-richtext"
    );
    const description = descEl ? descEl.textContent.trim().slice(0, 500) : "";

    // Utiliser le prix MTOP si disponible
    const mtopPrice = getMtopSkuPrice();
    if (mtopPrice) price = mtopPrice;

    // Extraire product_id depuis l'URL
    const idMatch = window.location.pathname.match(/\/item\/(\d+)\.html/);
    const productId = idMatch ? idMatch[1] : "";

    // SKU ID
    const skuAttr = getSelectedSkuAttr();
    const skuId = getSkuId(null);

    return {
      title,
      price,
      currency,
      url: normalizeAliExpressUrl(skuId),
      product_id: productId,
      sku_id: skuId || skuAttr,
      image_url: imageUrl,
      images: images.join("|"),
      rating,
      reviews_count: reviews.replace(/[^\d]/g, ""),
      description,
      category: "",
      brand: "",
      seller: "",
      shipping: "",
      availability: "",
      variants_count: "0",
    };
  }

  // ============================================
  // Amazon
  // ============================================
  function extractAmazon() {
    const url = window.location.href.split("?")[0];

    // Extraire ASIN depuis l'URL
    const asinMatch = window.location.pathname.match(/\/dp\/([A-Z0-9]{10})/i);
    const productId = asinMatch ? asinMatch[1] : "";

    const title = getText("#productTitle");

    let price =
      getText(".a-price .a-offscreen") ||
      getText("#priceblock_ourprice") ||
      getText("#priceblock_dealprice") ||
      getText(".a-price-whole");
    price = price.replace(/[^\d.,]/g, "");

    const mainImage =
      getAttr("#landingImage", "src") ||
      getAttr("#imgBlkFront", "src") ||
      getAttr("#main-image", "src");

    const thumbs = document.querySelectorAll(
      "#altImages .a-button-thumbnail img, .imageThumbnail img"
    );
    const images = Array.from(thumbs)
      .map((img) => (img.src || "").replace(/\._[A-Z]+\d+_\./, "."))
      .filter((src) => src && !src.includes("data:"));

    const ratingText = getText("#acrPopover .a-icon-alt") || getText(".a-icon-alt");
    const ratingNum = (ratingText.match(/[\d.,]+/) || [""])[0];

    const reviews = getText("#acrCustomerReviewText") || getText("#ratings-summary .a-size-base");

    const description =
      getText("#productDescription p") ||
      getText("#feature-bullets .a-list-item") ||
      "";

    let brand = getText("#bylineInfo") || getText(".po-brand .a-span9 .a-size-base");
    brand = brand.replace(/^(Marque|Brand|Visit the|Visiter)\s*:?\s*/i, "");

    const breadcrumbs = document.querySelectorAll("#wayfinding-breadcrumbs_feature_div li a");
    const category = breadcrumbs.length > 0
      ? breadcrumbs[breadcrumbs.length - 1].textContent.trim()
      : "";

    return {
      title,
      price,
      currency: getText(".a-price-symbol") || "USD",
      url,
      product_id: productId,
      sku_id: productId,
      image_url: mainImage,
      images: [mainImage, ...images].filter(Boolean).join("|"),
      rating: ratingNum,
      reviews_count: (reviews.match(/[\d,]+/) || [""])[0].replace(/,/g, ""),
      description: description.slice(0, 500),
      category,
      brand,
      seller: getText("#merchant-info a") || getText("#sellerProfileTriggerId"),
      shipping: getText("#deliveryMessageMirId .a-text-bold") || "",
      availability: getText("#availability span"),
      variants_count: String(
        document.querySelectorAll("#twister .a-button-text").length || 0
      ),
    };
  }
})();
