/**
 * MTOP Interceptor - S'execute dans le MAIN world (contexte de la page)
 * a document_start pour intercepter les reponses MTOP API AliExpress.
 *
 * Intercepte:
 * 1. fetch() - pour les appels same-origin
 * 2. XMLHttpRequest - pour les XHR classiques
 * 3. JSONP (script tags) - pour les appels cross-domain vers acs.aliexpress.com
 *
 * Les donnees sont stockees sur <html> via data-attributes pour etre
 * lues par detector.js (content script dans le monde isole).
 */
(function () {
  // Stocker les donnees MTOP sur <html> pour le content script
  function storeMtopData(data) {
    try {
      var skuData = data.data || data;
      var selectedSkuId = skuData.selectedSkuId || "";
      var priceMap = skuData.skuIdStrPriceInfoMap || null;

      // Prix de la variante selectionnee
      var selectedPrice = "";
      if (selectedSkuId && priceMap && priceMap[String(selectedSkuId)]) {
        var info = priceMap[String(selectedSkuId)];
        selectedPrice =
          info.actSkuCalPrice ||
          info.skuCalPrice ||
          info.skuAmount ||
          "";
        if (typeof selectedPrice === "object") {
          selectedPrice =
            selectedPrice.value || selectedPrice.formatedAmount || "";
        }
      }

      if (selectedSkuId) {
        document.documentElement.setAttribute(
          "data-dropping-sku-id",
          String(selectedSkuId)
        );
      }
      if (selectedPrice) {
        document.documentElement.setAttribute(
          "data-dropping-sku-price",
          String(selectedPrice)
        );
      }
      if (priceMap) {
        document.documentElement.setAttribute(
          "data-dropping-sku-map",
          JSON.stringify(Object.keys(priceMap))
        );
      }
    } catch (e) {
      /* silent */
    }
  }

  // Detecter si une URL est un appel MTOP pdp (adjust ou init)
  function isMtopPdp(url) {
    if (typeof url !== "string") return false;
    // Matcher les appels MTOP pdp: pc.adjust, pdp.recommend, etc.
    return (
      url.indexOf("mtop.aliexpress.pdp") !== -1 ||
      (url.indexOf("pdp") !== -1 && url.indexOf("adjust") !== -1) ||
      (url.indexOf("/fn/") !== -1 && url.indexOf("Detail") !== -1)
    );
  }

  // Extraire les donnees SKU depuis n'importe quel objet (recherche recursive)
  function extractSkuData(obj) {
    if (!obj || typeof obj !== "object") return;
    // Cas direct: l'objet contient selectedSkuId
    if (obj.selectedSkuId || obj.skuIdStrPriceInfoMap) {
      storeMtopData(obj);
      return;
    }
    // Chercher dans .data (structure MTOP standard)
    if (obj.data && typeof obj.data === "object") {
      if (obj.data.selectedSkuId || obj.data.skuIdStrPriceInfoMap) {
        storeMtopData(obj.data);
        return;
      }
    }
  }

  // Parser une reponse MTOP (JSON ou JSONP: mtopjsonpN({...}))
  function parseMtopResponse(text) {
    try {
      var jsonStr = text;
      var cbMatch = text.match(/^[^(]*\((.*)\);?\s*$/s);
      if (cbMatch) jsonStr = cbMatch[1];
      return JSON.parse(jsonStr);
    } catch (e) {
      return null;
    }
  }

  // ============================
  // 1. Intercepter fetch()
  // ============================
  var origFetch = window.fetch;
  window.fetch = function () {
    var url = arguments[0];
    if (typeof url === "object" && url.url) url = url.url;
    if (typeof url === "string" && isMtopPdp(url)) {
      return origFetch.apply(this, arguments).then(function (response) {
        var cloned = response.clone();
        cloned.text().then(function (text) {
          var parsed = parseMtopResponse(text);
          if (parsed) extractSkuData(parsed);
        });
        return response;
      });
    }
    return origFetch.apply(this, arguments);
  };

  // ============================
  // 2. Intercepter XMLHttpRequest
  // ============================
  var origOpen = XMLHttpRequest.prototype.open;
  var origSend = XMLHttpRequest.prototype.send;

  XMLHttpRequest.prototype.open = function (method, url) {
    this._droppingUrl = url || "";
    return origOpen.apply(this, arguments);
  };

  XMLHttpRequest.prototype.send = function () {
    var self = this;
    if (typeof self._droppingUrl === "string" && isMtopPdp(self._droppingUrl)) {
      self.addEventListener("load", function () {
        try {
          var text = self.responseText || "";
          var parsed = parseMtopResponse(text);
          if (parsed) extractSkuData(parsed);
        } catch (e) {
          /* silent */
        }
      });
    }
    return origSend.apply(this, arguments);
  };

  // ============================
  // 3. Intercepter JSONP (script tags)
  // AliExpress utilise souvent des script tags pour les appels MTOP cross-domain:
  //   <script src="https://acs.aliexpress.com/h5/mtop.aliexpress.pdp.pc.adjust/...?callback=mtopjsonp3">
  // On intercepte la creation de script tags pour wrapper le callback JSONP.
  // ============================
  var origCreateElement = Document.prototype.createElement;
  Document.prototype.createElement = function () {
    var el = origCreateElement.apply(this, arguments);
    var tagName = arguments[0];
    if (typeof tagName === "string" && tagName.toLowerCase() === "script") {
      // Surveiller quand le src est defini sur ce script
      var srcDescriptor = Object.getOwnPropertyDescriptor(
        HTMLScriptElement.prototype,
        "src"
      );
      if (srcDescriptor && srcDescriptor.set) {
        var origSrcSet = srcDescriptor.set;
        var origSrcGet = srcDescriptor.get;
        Object.defineProperty(el, "src", {
          set: function (val) {
            if (typeof val === "string" && isMtopPdp(val)) {
              // Extraire le nom du callback JSONP depuis l'URL
              var cbMatch = val.match(/[?&]callback=([^&]+)/);
              if (cbMatch) {
                var cbName = cbMatch[1];
                // Wrapper le callback JSONP pour capturer les donnees
                wrapJsonpCallback(cbName);
              }
            }
            return origSrcSet.call(this, val);
          },
          get: function () {
            return origSrcGet ? origSrcGet.call(this) : "";
          },
          configurable: true,
        });
      }
    }
    return el;
  };

  // Wrapper un callback JSONP: on remplace window[cbName] par une version
  // qui capture les donnees MTOP avant de transmettre au callback original.
  function wrapJsonpCallback(cbName) {
    // Le callback peut ne pas encore etre defini, on utilise defineProperty
    var origCb = window[cbName];
    if (typeof origCb === "function" && origCb._droppingWrapped) return;

    var wrapped = function (data) {
      try {
        extractSkuData(data);
      } catch (e) {
        /* silent */
      }
      // Appeler le callback original
      if (typeof origCb === "function") {
        return origCb.apply(this, arguments);
      }
    };
    wrapped._droppingWrapped = true;
    window[cbName] = wrapped;

    // Si le callback n'est pas encore defini, surveiller sa definition
    if (typeof origCb !== "function") {
      var currentVal = window[cbName];
      try {
        Object.defineProperty(window, cbName, {
          get: function () {
            return wrapped;
          },
          set: function (newVal) {
            if (typeof newVal === "function" && !newVal._droppingWrapped) {
              origCb = newVal;
              // Re-wrapper avec le nouveau callback
              wrapped = function (data) {
                try {
                  extractSkuData(data);
                } catch (e) {
                  /* silent */
                }
                return newVal.apply(this, arguments);
              };
              wrapped._droppingWrapped = true;
            }
          },
          configurable: true,
        });
      } catch (e) {
        // Si defineProperty echoue, on garde le wrapper simple
      }
    }
  }
})();
