/**
 * Scanner overlay - injected into the active tab.
 * Darkens the screen, lets user select QR code area with mouse,
 * decodes QR and sends result to background service worker.
 */

(function () {
  // Prevent double injection
  if (document.getElementById("dropping-scanner-overlay")) return;

  let screenshotUrl = null;
  let screenshotImg = null;

  // Listen for screenshot data from service worker
  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    if (msg.type === "SHOW_SCANNER" && msg.dataUrl) {
      screenshotUrl = msg.dataUrl;
      loadScreenshot();
      sendResponse({ ok: true });
    }
  });

  function loadScreenshot() {
    screenshotImg = new Image();
    screenshotImg.onload = () => initOverlay();
    screenshotImg.src = screenshotUrl;
  }

  function initOverlay() {
    const dpr = window.devicePixelRatio || 1;
    const vw = window.innerWidth;
    const vh = window.innerHeight;
    const imgW = screenshotImg.naturalWidth;
    const imgH = screenshotImg.naturalHeight;

    // Container
    const container = document.createElement("div");
    container.id = "dropping-scanner-overlay";
    container.style.cssText =
      "position:fixed;inset:0;z-index:2147483647;cursor:crosshair;user-select:none;";

    // Canvas
    const canvas = document.createElement("canvas");
    canvas.width = imgW;
    canvas.height = imgH;
    canvas.style.cssText = "width:100vw;height:100vh;display:block;";
    container.appendChild(canvas);

    const ctx = canvas.getContext("2d");

    // Scale factors: CSS pixels -> image pixels
    const scaleX = imgW / vw;
    const scaleY = imgH / vh;

    // Instructions
    const instructions = document.createElement("div");
    instructions.textContent = "Selectionnez la zone du QR code";
    instructions.style.cssText =
      "position:absolute;top:24px;left:50%;transform:translateX(-50%);" +
      "background:rgba(0,0,0,0.85);color:#fff;padding:10px 24px;" +
      "border-radius:8px;font-family:-apple-system,BlinkMacSystemFont,sans-serif;" +
      "font-size:14px;pointer-events:none;white-space:nowrap;";
    container.appendChild(instructions);

    // Escape hint
    const hint = document.createElement("div");
    hint.textContent = "Echap pour annuler";
    hint.style.cssText =
      "position:absolute;bottom:24px;left:50%;transform:translateX(-50%);" +
      "background:rgba(0,0,0,0.6);color:#71717a;padding:8px 16px;" +
      "border-radius:6px;font-family:-apple-system,BlinkMacSystemFont,sans-serif;" +
      "font-size:12px;pointer-events:none;";
    container.appendChild(hint);

    document.body.appendChild(container);

    // Draw darkened screenshot
    function draw(selRect) {
      ctx.clearRect(0, 0, imgW, imgH);

      // Draw screenshot
      ctx.drawImage(screenshotImg, 0, 0, imgW, imgH);

      // Dark overlay
      ctx.fillStyle = "rgba(0, 0, 0, 0.55)";
      ctx.fillRect(0, 0, imgW, imgH);

      if (selRect && selRect.w > 2 && selRect.h > 2) {
        // Clear selection area (show original image through)
        const sx = selRect.x * scaleX;
        const sy = selRect.y * scaleY;
        const sw = selRect.w * scaleX;
        const sh = selRect.h * scaleY;

        ctx.save();
        ctx.beginPath();
        ctx.rect(sx, sy, sw, sh);
        ctx.clip();
        ctx.drawImage(screenshotImg, 0, 0, imgW, imgH);
        ctx.restore();

        // Border
        ctx.strokeStyle = "#7c3aed";
        ctx.lineWidth = 2 * dpr;
        ctx.strokeRect(sx, sy, sw, sh);
      }
    }

    draw(null);

    // Mouse selection
    let startX = 0;
    let startY = 0;
    let selecting = false;

    container.addEventListener("mousedown", (e) => {
      startX = e.clientX;
      startY = e.clientY;
      selecting = true;
      e.preventDefault();
    });

    container.addEventListener("mousemove", (e) => {
      if (!selecting) return;
      draw({
        x: Math.min(startX, e.clientX),
        y: Math.min(startY, e.clientY),
        w: Math.abs(e.clientX - startX),
        h: Math.abs(e.clientY - startY),
      });
    });

    container.addEventListener("mouseup", async (e) => {
      if (!selecting) return;
      selecting = false;

      const rect = {
        x: Math.min(startX, e.clientX),
        y: Math.min(startY, e.clientY),
        w: Math.abs(e.clientX - startX),
        h: Math.abs(e.clientY - startY),
      };

      if (rect.w < 10 || rect.h < 10) return;

      instructions.textContent = "Analyse du QR code...";

      // Crop selection from original screenshot
      const sx = Math.round(rect.x * scaleX);
      const sy = Math.round(rect.y * scaleY);
      const sw = Math.round(rect.w * scaleX);
      const sh = Math.round(rect.h * scaleY);

      const cropCanvas = document.createElement("canvas");
      cropCanvas.width = sw;
      cropCanvas.height = sh;
      const cropCtx = cropCanvas.getContext("2d");
      cropCtx.drawImage(screenshotImg, sx, sy, sw, sh, 0, 0, sw, sh);

      // Decode QR with jsQR
      try {
        const imageData = cropCtx.getImageData(0, 0, sw, sh);
        const code = jsQR(imageData.data, imageData.width, imageData.height);

        if (!code) {
          instructions.textContent =
            "Aucun QR code detecte. Reessayez avec une selection plus precise.";
          instructions.style.color = "#fca5a5";
          return;
        }

        const qrData = code.data;
        instructions.textContent = "QR code detecte, connexion...";
        instructions.style.color = "#c4b5fd";

        // Send to background for connection
        const result = await chrome.runtime.sendMessage({
          type: "QR_SCANNED",
          data: qrData,
        });

        if (result && result.success) {
          instructions.textContent = "Connecte !";
          instructions.style.color = "#86efac";
          setTimeout(cleanup, 1000);
        } else {
          instructions.textContent =
            "Erreur: " + (result.error || "Connexion echouee");
          instructions.style.color = "#fca5a5";
        }
      } catch (err) {
        instructions.textContent = "Erreur: " + err.message;
        instructions.style.color = "#fca5a5";
      }
    });

    // Escape to cancel
    function onKeydown(e) {
      if (e.key === "Escape") {
        cleanup();
      }
    }
    document.addEventListener("keydown", onKeydown);

    function cleanup() {
      document.removeEventListener("keydown", onKeydown);
      container.remove();
    }
  }
})();
