/**
 * Client API pour communiquer avec le backend Dropping.
 */

const STORAGE_KEY = "dropping_session";

/**
 * Recuperer la session stockee.
 * @returns {Promise<{url: string, token: string} | null>}
 */
export async function getSession() {
  const data = await chrome.storage.local.get(STORAGE_KEY);
  return data[STORAGE_KEY] || null;
}

/**
 * Sauvegarder la session.
 * @param {string} url - URL du serveur
 * @param {string} token - Token de session
 */
export async function saveSession(url, token) {
  await chrome.storage.local.set({
    [STORAGE_KEY]: { url, token },
  });
}

/**
 * Supprimer la session.
 */
export async function clearSession() {
  await chrome.storage.local.remove(STORAGE_KEY);
}

/**
 * Connecter l'extension au serveur avec un token.
 * @param {string} serverUrl
 * @param {string} token
 * @returns {Promise<{status: string, csv_file_id: string}>}
 */
export async function connectToServer(serverUrl, token) {
  const response = await fetch(`${serverUrl}/api/extensions/connect`, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({ token }),
  });

  if (!response.ok) {
    const error = await response.json().catch(() => ({}));
    throw new Error(error.detail || `HTTP ${response.status}`);
  }

  return response.json();
}

/**
 * Envoyer les donnees d'un produit scrape.
 * @param {Record<string, string>} productData
 * @returns {Promise<{success: boolean, row_count: number, product_title: string}>}
 */
export async function sendProductData(productData) {
  const session = await getSession();
  if (!session) throw new Error("Not connected");

  const response = await fetch(`${session.url}/api/extensions/scrape`, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({
      token: session.token,
      product_data: productData,
    }),
  });

  if (!response.ok) {
    const error = await response.json().catch(() => ({}));
    throw new Error(error.detail || `HTTP ${response.status}`);
  }

  return response.json();
}

/**
 * Verifier le statut de la session.
 * @returns {Promise<{status: string, csv_file_id: string | null}>}
 */
export async function checkSessionStatus() {
  const session = await getSession();
  if (!session) return null;

  try {
    const response = await fetch(
      `${session.url}/api/extensions/sessions/${session.token}`
    );
    if (!response.ok) return null;
    return response.json();
  } catch {
    return null;
  }
}
