/**
 * Popup script - QR scan (overlay on page) + manual connection + status.
 */

const connectSection = document.getElementById("connect-section");
const connectedSection = document.getElementById("connected-section");
const statusBadge = document.getElementById("status-badge");
const scanBtn = document.getElementById("scan-btn");
const scanStatus = document.getElementById("scan-status");
const tokenInput = document.getElementById("token-input");
const urlInput = document.getElementById("url-input");
const connectBtn = document.getElementById("connect-btn");
const connectError = document.getElementById("connect-error");
const disconnectBtn = document.getElementById("disconnect-btn");
const serverUrlEl = document.getElementById("server-url");
const productsCountEl = document.getElementById("products-count");

// Au chargement, verifier le statut
document.addEventListener("DOMContentLoaded", checkStatus);

// Scanner QR - envoie au background qui injecte l'overlay sur la page
scanBtn.addEventListener("click", () => {
  chrome.runtime.sendMessage({ type: "START_SCAN" });
  window.close();
});

// Connexion manuelle
connectBtn.addEventListener("click", handleManualConnect);
tokenInput.addEventListener("keydown", (e) => {
  if (e.key === "Enter") handleManualConnect();
});

// Deconnexion
disconnectBtn.addEventListener("click", handleDisconnect);


// ============================================
// Connexion manuelle
// ============================================
async function handleManualConnect() {
  const token = tokenInput.value.trim();
  const url = urlInput.value.trim();

  if (!token) {
    showError("Veuillez entrer un token");
    return;
  }
  if (!url) {
    showError("Veuillez entrer l'URL du serveur");
    return;
  }

  connectBtn.disabled = true;
  connectBtn.textContent = "Connexion...";
  connectError.style.display = "none";

  try {
    const result = await chrome.runtime.sendMessage({
      type: "CONNECT",
      url: url,
      token: token,
    });

    if (result.success) {
      showConnected(url, 0);
    } else {
      showError(result.error || "Connexion echouee");
    }
  } catch (err) {
    showError(err.message || "Erreur de connexion");
  } finally {
    connectBtn.disabled = false;
    connectBtn.textContent = "Connecter";
  }
}


// ============================================
// Statut
// ============================================
async function checkStatus() {
  try {
    const status = await chrome.runtime.sendMessage({ type: "GET_STATUS" });
    const count = await chrome.runtime.sendMessage({
      type: "GET_PRODUCTS_COUNT",
    });

    if (status && status.connected) {
      showConnected(status.url, count.count || 0);
    } else {
      showDisconnected();
    }
  } catch {
    showDisconnected();
  }
}

async function handleDisconnect() {
  try {
    await chrome.runtime.sendMessage({ type: "DISCONNECT" });
  } catch {
    // Ignore
  }
  showDisconnected();
}

function showConnected(url, count) {
  connectSection.style.display = "none";
  connectedSection.style.display = "block";
  statusBadge.textContent = "Connecte";
  statusBadge.className = "badge badge-connected";
  serverUrlEl.textContent = url;
  productsCountEl.textContent = String(count);
  scanStatus.style.display = "none";
}

function showDisconnected() {
  connectSection.style.display = "block";
  connectedSection.style.display = "none";
  statusBadge.textContent = "Deconnecte";
  statusBadge.className = "badge badge-disconnected";
}

function showError(message) {
  connectError.textContent = message;
  connectError.style.display = "block";
}
