/**
 * Scanner page - opens in a full tab where camera access works.
 */

const cameraFeed = document.getElementById("camera-feed");
const scanStatus = document.getElementById("scan-status");
const scannerArea = document.getElementById("scanner-area");
const successArea = document.getElementById("success-area");
const cancelBtn = document.getElementById("cancel-btn");

let mediaStream = null;
let scanInterval = null;

cancelBtn.addEventListener("click", () => {
  stopScanner();
  window.close();
});

// Start scanning immediately
startScanner();

async function startScanner() {
  showStatus("Activation de la camera...", "scanning");

  try {
    mediaStream = await navigator.mediaDevices.getUserMedia({
      video: { facingMode: "environment", width: 640, height: 480 },
    });
    cameraFeed.srcObject = mediaStream;
    await cameraFeed.play();

    showStatus("Pointez vers le QR code", "scanning");

    if ("BarcodeDetector" in window) {
      const detector = new BarcodeDetector({ formats: ["qr_code"] });
      scanInterval = setInterval(() => detectQR(detector), 300);
    } else {
      showStatus(
        "BarcodeDetector non disponible dans ce navigateur. Utilisez le token manuel dans le popup.",
        "error"
      );
    }
  } catch (err) {
    if (err.name === "NotAllowedError") {
      showStatus(
        "Acces camera refuse. Cliquez sur l'icone camera dans la barre d'adresse pour autoriser.",
        "error"
      );
    } else {
      showStatus("Impossible d'acceder a la camera: " + err.message, "error");
    }
  }
}

async function detectQR(detector) {
  if (!cameraFeed.videoWidth) return;

  try {
    const barcodes = await detector.detect(cameraFeed);
    if (barcodes.length > 0) {
      const raw = barcodes[0].rawValue;
      stopScanner();
      handleQRData(raw);
    }
  } catch {
    // Ignore frame-by-frame detection errors
  }
}

function stopScanner() {
  if (scanInterval) {
    clearInterval(scanInterval);
    scanInterval = null;
  }
  if (mediaStream) {
    mediaStream.getTracks().forEach((t) => t.stop());
    mediaStream = null;
  }
  cameraFeed.srcObject = null;
}

async function handleQRData(raw) {
  showStatus("QR code detecte, connexion...", "scanning");

  let url, token;
  try {
    const data = JSON.parse(raw);
    url = data.url;
    token = data.token;
  } catch {
    showStatus("QR code invalide (format JSON attendu)", "error");
    return;
  }

  if (!url || !token) {
    showStatus("QR code incomplet (url et token requis)", "error");
    return;
  }

  try {
    const result = await chrome.runtime.sendMessage({
      type: "CONNECT",
      url: url,
      token: token,
    });

    if (result.success) {
      // Show success and close tab
      scannerArea.style.display = "none";
      successArea.style.display = "block";
      cancelBtn.style.display = "none";
      setTimeout(() => window.close(), 1500);
    } else {
      showStatus(
        "Erreur: " + (result.error || "Connexion echouee"),
        "error"
      );
    }
  } catch (err) {
    showStatus("Erreur: " + err.message, "error");
  }
}

function showStatus(message, type) {
  scanStatus.textContent = message;
  scanStatus.className = type;
}
